﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using SdlDotNet.Graphics;
using System.Drawing;
using SdlDotNet.Core;
using SdlDotNet.Input;

namespace Pong
{
    /// <summary>
    /// Die Klasse, die sich um die Verarbeitung des Spiels an sich kümmert
    /// </summary>
    class PongMaster
    {
        #region Variablendeklarierung

        /// <summary>
        /// Array der verwendeten Pongitems
        /// </summary>
        PongItem[] items;

        /// <summary>
        /// Legt den aktuellen Status des PongMasters fest
        /// </summary>
        Status stat;

        /// <summary>
        /// Delegate für das Spielbeenden
        /// </summary>
        public delegate void SpielBeender();

        /// <summary>
        /// Event für das Spielbeenden
        /// </summary>
        public event SpielBeender spielbeender;

        /// <summary>
        /// DialogMenu zum Abfragen, ob das Spiel nach Esc-Tastendruck wirklich beendet werden soll
        /// </summary>
        DialogMenu beendabfrage;

        /// <summary>
        /// DialogMenu zum Abfragen, ob nach dem Ende des Spiels nochmal ein gleichgeartetes Spiel gespielt werden soll
        /// </summary>
        public DialogMenu nochmalabfrage;

        /// <summary>
        /// Die vom Anwender eingestellten Optionen
        /// </summary>
        Option[] optionen;

        /// <summary>
        /// Legt fest, ob gerade ein EventItem im Spiel ist
        /// </summary>
        bool eventitemaktiv;

        /// <summary>
        /// Legt fest, wie viele Frames seit dem letzten Treffen eines EventItems vergangen sind
        /// </summary>
        int framesseitletztevent = 0;

        /// <summary>
        /// Legt fest, wie viele Frames zwischen dem Treffen eines EventItems und dem Neuerscheinen eines weiteren EventItems vergehen
        /// </summary>
        int frameszwievents;

        /// <summary>
        /// Alle im Spiel verwendeten EventItems
        /// </summary>
        EventItem[] eventitems;

        /// <summary>
        /// Legt den Index, des momentan aktiven EventItems fest
        /// </summary>
        int aktivevent;

        /// <summary>
        /// Zufallszahlengenerator für das Auswählen der EventItems
        /// </summary>
        Random random;

        /// <summary>
        /// Legt fest, ob generell EventItems im Spiel auftreten dürfen
        /// </summary>
        bool eventsverwenden;

        /// <summary>
        /// Legt fest, ob das linke Paddle computergesteuert ist
        /// </summary>
        bool kiaktiv;

        #endregion

        #region Konstruktor

        /// <summary>
        /// Erstellt einen PongMaster mit den eingestellten Optionen optionen und den EventItems eventitems
        /// </summary>
        /// <param name="optionen"></param>
        /// <param name="eventitems"></param>
        public PongMaster(Option[] optionen,EventItem[] eventitems)
        {
            this.optionen = optionen;
            this.eventitems = eventitems;

            random = new Random();

            // Initialisieren der DialogMenus
            beendabfrage = new DialogMenu("Willst du das Spiel wirklich beenden?");
            beendabfrage.abgelehnt += new DialogMenu.Abgelehnt(weiterspielen);

            nochmalabfrage = new DialogMenu("Soll nocheinmal gespielt werden?");
            nochmalabfrage.akzeptiert += new DialogMenu.Akzeptiert(init);
        }

        #endregion

        #region Variablenzugriff

        /// <summary>
        /// Setzt die PongItems des Pongmaster oder gibt sie zurück
        /// </summary>
        public PongItem[] Items
        {
            get { return items; }
            set { items = value; }
        }

        /// <summary>
        /// Setzt den Status des Pongmaster oder gibt diesen zurück
        /// </summary>
        public Status Stat
        {
            get { return stat; }
            set { stat = value; }
        }

        /// <summary>
        /// Setzt die Optionen oder gibt diese zurück
        /// </summary>
        public Option[] Optionen
        {
            get { return optionen; }
            set { optionen = value; }
        }

        /// <summary>
        /// Setzt, ob ein Eventitem aktiv ist, oder gibt dies zurück
        /// </summary>
        public bool Eventitemaktiv
        {
            get { return eventitemaktiv; }
            set { eventitemaktiv = value; }
        }

        #endregion

        #region Methoden

        /// <summary>
        /// Wird beim Zeichnen jedes Frames aufgerufen
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        public void Events_Tick(object sender, TickEventArgs e)
        {
            if (Stat == Status.Aktiv)
            {
                if (eventsverwenden)
                {
                    if (framesseitletztevent == frameszwievents)
                    {
                        // Zufällig ein EventItem aussuchen
                        aktivevent = random.Next(eventitems.Length);

                        // Überprüfen, ob es noch verwendbare EventItems gibt
                        bool wasverwendet = false;
                        foreach (EventItem ei in eventitems)
                        {
                            if (ei.Verwendet)
                            {
                                wasverwendet = true;
                                break;
                            }
                        }

                        // Nur EventItems aktivieren, wenn noch verwendbare EventItems existieren
                        if (!wasverwendet)
                            eventsverwenden = false;
                        else
                        {
                            // Nur Events, die verwendet werden dürfen, können aktiviert werden
                            while (!eventitems[aktivevent].Verwendet)
                                aktivevent = random.Next(eventitems.Length);

                            // EventItem aktivieren
                            eventitems[aktivevent].Aktiv = true;
                            framesseitletztevent = 0;

                            Eventitemaktiv = true;
                        }
                    }

                    if (!Eventitemaktiv)
                        framesseitletztevent += 1;
                }

                if (kiaktiv)
                {
                    foreach (PongItem pitem in Items)
                    {
                        if (pitem is Spieler && (pitem as Spieler).Seit == Spieler.Seite.Links)
                        {
                            // Paddle soll sich in die Richtung des am nächsten gelegenen Balls bewegen, der in Richtung des Paddles fliegt

                            Ball ball = null;
                           
                            foreach (PongItem pi in Items)
                            {
                                if (pi is Ball && (pi as Ball).BallSpeedX < 0)
                                {
                                    if (ball == null || ball.X > pi.X)
                                        ball = pi as Ball;
                                }
                            }

                            if (ball != null)
                            {
                                if ((ball.Y + ball.Height / 2) > (pitem.Y + pitem.Height / 2))
                                {
                                    paddleRunter(Spieler.Seite.Links, true);
                                    paddleRauf(Spieler.Seite.Links, false);
                                }
                                else
                                {
                                    paddleRunter(Spieler.Seite.Links, false);
                                    paddleRauf(Spieler.Seite.Links, true);
                                }
                            }
                            else
                            {
                                paddleRunter(Spieler.Seite.Links, false);
                                paddleRauf(Spieler.Seite.Links, false);
                            }
                        }
                    }
                }

                draw();
                Video.Screen.Update();
            }
            else if (Stat == Status.BeendAbfrage)
            {
                beendabfrage.Events_Tick(sender, e);
            }
            else if (Stat == Status.NochmalAbfrage)
            {
                nochmalabfrage.Events_Tick(sender, e);
            }
        }


        /// <summary>
        /// Wird bei jedem Drücken einer Taste aufgerufen
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        public void Events_Keydown(object sender, KeyboardEventArgs e)
        {
            if (Stat == Status.Aktiv)
            {
                // Nur, wenn ohne KI gespielt wird, kann das linke Paddle bewegt werden
                if (!kiaktiv)
                {
                    // A fährt das linke Paddle hoch
                    if (e.Key == Key.A)
                    {
                        paddleRauf(Spieler.Seite.Links, true);
                    }
                    // Y fährt das linke Paddle runter
                    else if (e.Key == Key.Z)
                    {
                        paddleRunter(Spieler.Seite.Links, true);
                    }
                }

                // Pfeil hoch fährt das rechts Paddle hoch
                if (e.Key == Key.UpArrow)
                {
                    paddleRauf(Spieler.Seite.Rechts, true);
                }
                // Pfeil runter fährt das rechte Paddle runter
                else if (e.Key == Key.DownArrow)
                {
                    paddleRunter(Spieler.Seite.Rechts, true);
                }
                // Escape beendet das aktuelle Spiel
                else if (e.Key == Key.Escape)
                {
                    Stat = Status.BeendAbfrage;
                }
            }
            else if (Stat == Status.BeendAbfrage)
            {
                beendabfrage.Events_Keydown(sender, e);
            }
            else if (Stat == Status.NochmalAbfrage)
            {
                nochmalabfrage.Events_Keydown(sender, e);
            }
        }


        /// <summary>
        /// Wird bei jedem Loslassen einer Taste aufgerufen
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        public void Events_Keyup(object sender, KeyboardEventArgs e)
        {
            if (Stat == Status.Aktiv)
            {
                // Nur, wenn ohne KI gespielt wird, kann das linke Paddle bewegt werden
                if (!kiaktiv)
                {
                    // Linkes Paddle soll nicht mehr hoch fahren
                    if (e.Key == Key.A)
                    {
                        paddleRauf(Spieler.Seite.Links, false);
                    }
                    // Linkes Paddle soll nicht mehr runter fahren
                    else if (e.Key == Key.Z)
                    {
                        paddleRunter(Spieler.Seite.Links, false);
                    }
                }
                
                // Rechtes Paddle soll nicht mehr hoch fahren
                if (e.Key == Key.UpArrow)
                {
                    paddleRauf(Spieler.Seite.Rechts, false);
                }
                // Rechtes Paddle soll nicht mehr runter fahren
                else if (e.Key == Key.DownArrow)
                {
                    paddleRunter(Spieler.Seite.Rechts, false);
                }
            }
        }

        /// <summary>
        /// Zeichnet alle PongItems des PongMasters
        /// </summary>
        public void draw()
        {
            // Erstmal alles wegmachen
            Video.Screen.Fill(Color.Black);

            // Pongitems an ihren Positonen zeichnen
            foreach (PongItem pongitem in Items)
            {
                if (pongitem!=null)
                    pongitem.draw();
            }
        }

        /// <summary>
        /// Legt für den Spieler auf der Seite seite die Variable rauf fest
        /// </summary>
        /// <param name="seite"></param>
        /// <param name="druck"></param>
        public void paddleRauf(Spieler.Seite seite, bool rauf)
        {
            foreach (PongItem pi in Items)
            {
                if (pi is Spieler && (pi as Spieler).Seit == seite)
                {
                    (pi as Spieler).Rauf = rauf;
                }
            }
        }

        /// <summary>
        /// Legt für den Spieler auf der Seite seite die Variable rauf fest
        /// </summary>
        /// <param name="seite"></param>
        /// <param name="druck"></param>
        public void paddleRunter(Spieler.Seite seite, bool runter)
        {
            foreach (PongItem pi in Items)
            {
                if (pi is Spieler && (pi as Spieler).Seit == seite)
                {
                    (pi as Spieler).Runter = runter;
                }
            }
        }

        /// <summary>
        /// Lässt das Spiel weitergehen
        /// </summary>
        public void weiterspielen()
        {
            Stat = Status.Aktiv;
        }

        /// <summary>
        /// Stellt die Frameanzahl zwischen dem Treffen eines EventItems und dem erneuten Auftreten eines EventItems ein
        /// </summary>
        private void framesEinstellen()
        {
            switch ((EventAuftreten)Optionen[(int)Pong.Optionen.EventAuftreten].Wert)
            {
                case EventAuftreten.Selten :
                    frameszwievents = 1200;
                    break;
                case EventAuftreten.Normal :
                    frameszwievents = 600;
                    break;
                case EventAuftreten.Oft :
                    frameszwievents = 300;
                    break;
            }
        }

        /// <summary>
        /// Stellt ein, ob generell EventItems im Spiel auftreten dürfen
        /// </summary>
        private void eventsEinstellen()
        {
            eventsverwenden = false;

            foreach (EventItem ei in eventitems)
            {
                if (ei.Verwendet)
                {
                    eventsverwenden = true;
                    break;
                }
            }
        }

        /// <summary>
        /// Stellt das Spiel je nach KIEinstellung ein
        /// </summary>
        private void KIeinstellen()
        {
            switch ((Pong.JaNein)Optionen[(int)Pong.Optionen.KIverwenden].Wert)
            {
                case Pong.JaNein.Ja :
                    kiaktiv = true;
                    break;
                case Pong.JaNein.Nein :
                    kiaktiv = false;
                    break;
            }
        }

        /// <summary>
        /// Initialisiert die nötigen PongItems
        /// </summary>
        public void init()
        {
            framesEinstellen();
            framesseitletztevent = 0;

            eventsEinstellen();

            // Anzahl der Bälle je nach eingestellten Optionen festlegen
            int anzbaelle = (int)Optionen[(int)Pong.Optionen.AnzahlBaelle].Wert;

            // Zusätzlicher Platz in den Items für den doppelten Ball
            int zusatz = 0;
            if (eventitems[(int)Pong.EventItems.BallVerdopplung].Verwendet)
                zusatz = 1;
                   
            // Instanzieren der benötigten PongItems
            Ball[] baelle = new Ball[anzbaelle];
            Spieler spieler1 = new Spieler(Spieler.Seite.Links);
            Spieler spieler2 = new Spieler(Spieler.Seite.Rechts);
            Punktestand punktestand = new Punktestand();

            // Die Balleinstellung je nach der ausgewählten Option festlegen
            int be = (int)Optionen[(int)Pong.Optionen.BallEinstellung].Wert;
            Ball.BallEinstellung balleinstellung = (Ball.BallEinstellung)Enum.ToObject(typeof(Ball.BallEinstellung), be);

            // Festlegen der Startpositionen der Bälle
            for (int i = 0; i < anzbaelle; i++)
            {
                if (i % 2 == 0)
                {
                    baelle[i] = new Ball(spieler1.Seit, balleinstellung);
                    baelle[i].X = spieler1.Width + 1;
                }
                else
                {
                    baelle[i] = new Ball(spieler2.Seit, balleinstellung);
                    baelle[i].X = Video.Screen.Width - spieler2.Width - baelle[i].Width - 1;
                }
                if (i < 2)
                {
                    baelle[i].Y = 0;
                }
                else
                {
                    baelle[i].Y = Video.Screen.Height - baelle[i].Height - 1;
                }
            }

            // Festlegen der Startpositionen der Paddle
            spieler1.Y = ((double)Video.Screen.Height / 2) - ((double)spieler1.Height / 2);
            spieler2.Y = ((double)Video.Screen.Height / 2) - ((double)spieler2.Height / 2);
            spieler2.X = Video.Screen.Width - spieler2.Width;
            
            // Festlegen der Position des Punktestands
            punktestand.X = Video.Screen.Width / 2;
            punktestand.Y = 0;
            punktestand.update(spieler1.Punkte, spieler2.Punkte);

            // Auffüllen des PongItem-Array
            Items = new PongItem[3 + anzbaelle + eventitems.Length + zusatz];
            Items[0] = spieler1;
            Items[1] = spieler2;
            Items[2] = punktestand;

            for (int i = 0; i < anzbaelle; i++)
            {
                Items[3 + i] = baelle[i];
            }

            for (int i = 0; i < eventitems.Length; i++)
            {
                Items[3 + anzbaelle + i] = eventitems[i];
            }

            // Weist den DialogMenuEvents die richtigen Methoden zu
            if (spielbeender != null)
            {
                beendabfrage.akzeptiert += new DialogMenu.Akzeptiert(spielbeender);
                nochmalabfrage.abgelehnt += new DialogMenu.Abgelehnt(spielbeender);
            }

            if (eventitems[(int)Pong.EventItems.BallVerdopplung].Verwendet)
            {
                Ball b = new Ball(Spieler.Seite.Links, balleinstellung);
                b.Aktiv = false;
                Items[Items.Length - 1] = b;
            }
            
            // weist allen PongItems als PongMaster die Referenz des aktuellen PongMaster zu
            foreach (PongItem pongitem in Items)
                pongitem.Pongmaster = this;

            // Überprüft, ob mit oder ohne KI gespielt werden soll
            KIeinstellen();

            // Lässt das Spiel laufen
            Stat = Status.Aktiv;
        }

        #endregion

        #region Enumeration Status

        /// <summary>
        /// Enumeration für die verschiedenen Stati des PongMasters
        /// </summary>
        public enum Status : int
        {
            /// <summary>
            /// Das Spiel läuft
            /// </summary>
            Aktiv = 0,

            /// <summary>
            /// Es wird gerade abgefragt, ob das Spiel beendet werden soll
            /// </summary>
            BeendAbfrage = 1,

            /// <summary>
            /// Ein Spieler hat gewonnen und es wird gerade abgefragt, ob noch ein Spiel mit den gleichen Einstellungen gespielt werden soll
            /// </summary>
            NochmalAbfrage = 2,
        }

        #endregion

        #region Enumeration EventAuftreten

        /// <summary>
        /// Enumeration für die Häufigkeit, in der EventItems im Spiel erscheinen
        /// </summary>
        public enum EventAuftreten : int
        {
            /// <summary>
            /// Seltenes Auftreten von EventItems
            /// </summary>
            Selten = 0,

            /// <summary>
            /// Normale Häufigkeit für das Auftreten von EventItems
            /// </summary>
            Normal = 1,

            /// <summary>
            /// EventItems erscheinen oft
            /// </summary>
            Oft = 2,
        }

        #endregion
    }
}
