﻿using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Drawing;
using System.Data;
using System.Text;
using System.Windows.Forms;
using System.Runtime.InteropServices;

///
/// WRITTEN BY FLORIAN RAPPL, 2011
/// CODE IS FREE TO USE -- CPOL [ Code Project Open License ]
/// 

namespace rsi.Controls.iToolTip
{
    public partial class ToolTipTextBox : UserControl
    {
        #region members

        string mCue;
        string _toolTip;
        InternetToolTip toolTip;

        #endregion

        #region constants

        const int EM_LINECOUNT = 0xBA;
        const int EM_LINEINDEX = 0xBB;
        const int EM_LINELENGTH = 0xC1;

        #endregion

        #region ctor

        /// <summary>
        /// Creates a new ToolTipTextBox control.
        /// </summary>
        public ToolTipTextBox()
        {
            InitializeComponent();
            help.Image = Resources.small_lightbulb;
            toolTip = new InternetToolTip();
            toolTip.DataProvider = new StringDataProvider();
            toolTip.DataView = new BasicToolTipView();
        }

        #endregion

        #region overrides and events

        protected override void OnSizeChanged(EventArgs e)
        {
            if (Height != textBox.Height)
                Height = textBox.Height;
        }

        protected override void OnHandleCreated(EventArgs e)
        {
            base.OnHandleCreated(e);
            updateCue();
        }

        #endregion

        #region dll-import

        [DllImport("user32.dll", EntryPoint = "SendMessageW")]
        private static extern IntPtr SendMessage(IntPtr hWnd, int msg, IntPtr wp, IntPtr lp);

        #endregion

        #region methods

        void updateCue()
        {
            if (!IsHandleCreated || string.IsNullOrEmpty(mCue))
                return;

            IntPtr mem = Marshal.StringToHGlobalUni(mCue);
            SendMessage(textBox.Handle, 0x1501, (IntPtr)1, mem);
            Marshal.FreeHGlobal(mem);
        }

        void textBox_SizeChanged(object sender, EventArgs e)
        {
            if (textBox.Height > Height)
                OnSizeChanged(e);
        }

        #endregion

        #region properties

        /// <summary>
        /// Gets or sets the textbox text.
        /// </summary>
        public override string Text
        {
            get
            {
                return textBox.Text;
            }
            set
            {
                textBox.Text = value;
            }
        }

        /// <summary>
        /// Gets or sets the loading text of the tooltip.
        /// </summary>
        public string LoadText
        {
            get
            {
                return toolTip.LoadText;
            }
            set
            {
                toolTip.LoadText = value;
            }
        }

        /// <summary>
        /// Gets the containing textbox.
        /// </summary>
        public TextBox TextBox
        {
            get
            {
                return textBox;
            }
        }

        /// <summary>
        /// Gets or sets the ToolTip to use with the lightbulb.
        /// </summary>
        public string ToolTip
        {
            get { return _toolTip; }
            set
            {
                _toolTip = value;
                toolTip.SetToolTip(help, value);
            }
        }

        /// <summary>
        /// Gets or sets the tooltip data to use.
        /// </summary>
        public IToolTipDataProvider ToolTipData
        {
            get { return toolTip.DataProvider; }
            set { toolTip.DataProvider = value; }
        }

        /// <summary>
        /// Gets or sets the tooltip view to use.
        /// </summary>
        public IToolTipView ToolTipView
        {
            get { return toolTip.DataView; }
            set { toolTip.DataView = value; }
        }

        /// <summary>
        /// Gets or sets the hint for the textbox.
        /// </summary>
        public string Cue
        {
            get
            {
                return mCue;
            }
            set
            {
                mCue = value;
                updateCue();
            }
        }

        /// <summary>
        /// Gets the number of lines of the textbox. 
        /// </summary>
        public int LineCount
        {
            get
            {
                Message msg = Message.Create(textBox.Handle, EM_LINECOUNT, IntPtr.Zero, IntPtr.Zero);
                base.DefWndProc(ref msg);
                return msg.Result.ToInt32();
            }
        }

        /// <summary>
        /// Gets the index of a certain line.
        /// </summary>
        /// <param name="index">Index position of the character.</param>
        /// <returns>Index of the corresponding line.</returns>
        public int LineIndex(int index)
        {
            Message msg = Message.Create(textBox.Handle, EM_LINEINDEX, (IntPtr)index, IntPtr.Zero);
            base.DefWndProc(ref msg);
            return msg.Result.ToInt32();
        }

        /// <summary>
        /// Gets the length of a certain line.
        /// </summary>
        /// <param name="index">Index position of the line.</param>
        /// <returns>Corresponding length of the line.</returns>
        public int LineLength(int index)
        {
            Message msg = Message.Create(textBox.Handle, EM_LINELENGTH, (IntPtr)index, IntPtr.Zero);
            base.DefWndProc(ref msg);
            return msg.Result.ToInt32();
        }

        #endregion
    }
}
