﻿using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Drawing;
using System.Text;
using System.Windows.Forms;
using System.Drawing.Drawing2D;
using System.Runtime.InteropServices;

///
/// WRITTEN BY FLORIAN RAPPL, 2011
/// CODE IS FREE TO USE -- CPOL [ Code Project Open License ]
/// 

namespace rsi.Controls.iToolTip
{
    public partial class BasicToolTipView : Form, IToolTipView
    {
        #region members

        int frames;
        double faderate;
        int sliderate;
        int slideremain;
        int ttw;
        Color gradientColorOne;
        Color gradientColorTwo;
        Color color;
        Color exceptionColor;
        Color loadColor;
        Bitmap errorImage;
        Bitmap loadImage;
        Effect showEffect;
        Timer effectTimer;
        int showEffectTime;
        DrawMode drawMode;
        object toolTip;

        #endregion

        #region constants

        const int SW_SHOWNOACTIVATE = 4;
        const int HWND_TOPMOST = -1;
        const uint SWP_NOACTIVATE = 0x0010;
        const int MS_PER_FRAME = 13;

        #endregion

        #region events

        /// <summary>
        /// Use this event in order to customize the drawing behavior of the basic tooltip view.
        /// </summary>
        public event DrawToolTipEventHandler DrawToolTipView;

        /// <summary>
        /// Use this event in combination with OwnerDrawVariable to adjust width / height of the view.
        /// </summary>
        public event MeasureToolTipEventHandler MeasureToolTipView;

        #endregion

        #region ctor

        /// <summary>
        /// Creates an instance the basic implementation of a ToolTipView
        /// </summary>
        public BasicToolTipView()
        {
            //Sets the standard effect values
            showEffectTime = 0;
            effectTimer = new Timer();
            effectTimer.Interval = MS_PER_FRAME;
            effectTimer.Tick += new EventHandler(effectTimer_Tick);
            showEffect = Effect.Appear;

            //Sets the standard maximum width
            ttw = 120;
            drawMode = DrawMode.Normal;
            
            //Sets the standard colors
            exceptionColor = Color.Red;
            loadColor = Color.Black;
            errorImage = Resources.error;
            gradientColorOne = Color.LightYellow;
            gradientColorTwo = Color.Orange;

            //Sets the standard loading image
            loadImage = Resources.default_load;

            //Builds the basic form
            InitializeComponent();
            SetStyle(ControlStyles.SupportsTransparentBackColor | ControlStyles.UserPaint | ControlStyles.AllPaintingInWmPaint, true);

            //Some properties for transparency
            BackColor = SystemColors.Control;
            panel.BackColor = SystemColors.Control;
            TransparencyKey = SystemColors.Control;

            //Workaround to not steal focus on first time show
            Show();
            Hide();
        }

        #endregion

        #region dll-import pinvoke

        /// <summary>
        /// Sets the window position for a top-most non-active window
        /// </summary>
        /// <param name="hWnd">window handle</param>
        /// <param name="hWndInsertAfter">placement-order handle</param>
        /// <param name="X">horizontal position</param>
        /// <param name="Y">vertical position</param>
        /// <param name="cx">width</param>
        /// <param name="cy">height</param>
        /// <param name="uFlags">window positioning flags</param>
        /// <returns>window is shown</returns>
        [DllImport("user32.dll", EntryPoint = "SetWindowPos")]
        static extern bool SetWindowPos(
             int hWnd, 
             int hWndInsertAfter,
             int X,
             int Y,
             int cx,
             int cy,
             uint uFlags);

        /// <summary>
        /// Shows a window
        /// </summary>
        /// <param name="hWnd">pointer to a form</param>
        /// <param name="nCmdShow">additional commands</param>
        /// <returns>success</returns>
        [DllImport("user32.dll")]
        static extern bool ShowWindow(IntPtr hWnd, int nCmdShow);

        #endregion

        #region IToolTipView Member

        public void DrawLoading(string text)
        {
            toolTip = text;
            loadimg.Image = loadImage;
            Draw(true, text, loadColor);
        }

        public void DrawToolTip(object tooltip)
        {
            toolTip = tooltip;
            Draw(false, tooltip.ToString(), ForeColor);
        }

        public void DrawException(Exception exception)
        {
            toolTip = exception;
            loadimg.Image = errorImage;
            Draw(true, exception.Message, exceptionColor);
        }

        public void ShowToolTip(Point p, Size sz)
        {
            //Sets the location
            Location = new Point(p.X + sz.Width / 2 - Width / 2, p.Y - Height);

            //Shows this window without activating it
            ShowWindow(Handle, SW_SHOWNOACTIVATE);


            //Sets the position and handles the window as topmost without activating the form
            SetWindowPos(Handle.ToInt32(), HWND_TOPMOST, Left, Top, Width, Height, SWP_NOACTIVATE);

            if (ShowEffect != Effect.Appear)
            {
                //Calculate basic framerate
                frames = showEffectTime / MS_PER_FRAME;

                //Too low?! Do nothing!
                if (frames <= 0)
                    return;

                //We have to set this for fading
                if (ShowEffect == Effect.Fade || ShowEffect == Effect.SlideFade)
                {
                    Opacity = 0.0;
                    faderate = 1.0 / (double)frames;
                }

                //We have to determine this for sliding
                if (ShowEffect == Effect.Slide || ShowEffect == Effect.SlideFade)
                {
                    sliderate = (Top + Height) / frames;
                    slideremain = (Top + Height) % frames;
                    Location = new Point(Left, -Height);
                }

                //Start the effect timer
                effectTimer.Start();
            }
        }

        public void HideToolTip()
        {
            //If Animation is still running - stop animation
            if (frames > 0)
                effectTimer.Stop();

            //Hides the tooltip
            Hide();
        }

        #endregion

        #region methods

        /// <summary>
        /// Sets the animation that is used for showing the tooltip
        /// </summary>
        /// <param name="effect">The effect that should be used</param>
        /// <param name="duration">The time in ms that should be used</param>
        public BasicToolTipView Animation(Effect effect, int duration)
        {
            //Sets the properties
            ShowEffectTime = duration;
            ShowEffect = effect;

            return this;
        }

        void effectTimer_Tick(object sender, EventArgs e)
        {
            //Calculate now Top-Coordinate
            int top = Top;

            if (slideremain > 0)
            {
                top += slideremain;
                slideremain = 0;
            }

            //Set new properties (Opacity and Top-Coordinate)
            top += sliderate;
            Location = Location + new Size(0, top - Top);
            Opacity += faderate;

            //Determine if we have to do more frames for the animation
            frames--;

            if (frames == 0)
                effectTimer.Stop();
        }

        void Draw(bool visible, string text, Color c)
        {
            //Sets the tooltip / exception / loading string (w or w/o loading image)
            Text = text;
            color = c;
            Size delta;

            if (drawMode == DrawMode.OwnerDrawVariable && MeasureToolTipView != null) //Here an event is required to determine the best size
            {
                MeasureToolTipEventArgs e = new MeasureToolTipEventArgs(Font, toolTip, Height, Width);
                MeasureToolTipView(this, e);
                delta = new Size(e.Width, e.Height) - Size;
            }
            else //Here the form will decide its best size
            {
                //If the image is visible we have to do more calculations
                if (visible)
                {
                    Size t = TextRenderer.MeasureText(text, Font);
                    delta = new Size(errorImage.Width + t.Width + 12, 16 + Math.Max(t.Height, errorImage.Height)) - Size;
                    loadimg.Size = loadimg.Image.Size;
                }
                else //Else we will just use all the space (and more) for the text
                    delta = TextRenderer.MeasureText(text, Font, new Size(ttw, 0), TextFormatFlags.WordBreak) + new Size(8, 16) - Size;
            }

            //Did something change from the measurement?!
            if(delta.Width != 0 || delta.Height != 0)
            {
                Size += delta;
                Location = new Point(Left - delta.Width / 2, Top - delta.Height);
            }

            //Now do the real work
            loadimg.Visible = visible;
            panel.Invalidate();
        }

        #endregion

        #region properties

        /// <summary>
        /// Gets or sets which drawmode to use - fixed uses height and width determined by the program, 
        /// variable gives you the control
        /// </summary>
        public DrawMode DrawMode
        {
            get { return drawMode; }
            set { drawMode = value; }
        }

        /// <summary>
        /// Gets or sets the first (top) gradient color
        /// </summary>
        public Color GradientColorOne
        {
            get { return gradientColorOne; }
            set { gradientColorOne = value; }
        }

        /// <summary>
        /// Gets or sets the second (bottom) gradient color
        /// </summary>
        public Color GradientColorTwo
        {
            get { return gradientColorTwo; }
            set { gradientColorTwo = value; }
        }

        /// <summary>
        /// Gets or sets the time in ms that is used for animating the tooltip
        /// </summary>
        public int ShowEffectTime
        {
            get { return showEffectTime; }
            set { showEffectTime = value; }
        }

        /// <summary>
        /// Gets or sets which effect should be used on showing the tooltip
        /// </summary>
        public Effect ShowEffect
        {
            get { return showEffect; }
            set { showEffect = value; }
        }

        /// <summary>
        /// Gets or sets the (maximum) width of the tooltip
        /// </summary>
        public int ToolTipWidth
        {
            get { return ttw; }
            set { ttw = value; }
        }

        /// <summary>
        /// Gets or sets the text color on loading the tooltip
        /// </summary>
        public Color LoadColor
        {
            get { return loadColor; }
            set { loadColor = value; }
        }

        /// <summary>
        /// Gets or sets the text color for showing exceptions
        /// </summary>
        public Color ExceptionColor
        {
            get { return exceptionColor; }
            set { exceptionColor = value; }
        }

        /// <summary>
        /// Gets or sets the image that will be shown on asynchronous requests.
        /// </summary>
        [Description("This image will be shown on asynchronous requests.")]
        public Bitmap LoadImage
        {
            get { return loadImage; }
            set { loadImage = value; }
        }

        /// <summary>
        /// Gets or sets the image that will be shown on errors during asynchronous requests.
        /// </summary>
        [Description("This image will be shown on errors during the execution.")]
        public Bitmap ErrorImage
        {
            get { return errorImage; }
            set { errorImage = value; }
        }

        #endregion

        #region paint-event

        void panel_Paint(object sender, PaintEventArgs e)
        {
            DrawToolTipEventArgs dte = new DrawToolTipEventArgs(e.Graphics, Font, new Rectangle(Point.Empty, Size), 
                                                                color, gradientColorOne, gradientColorTwo, toolTip);

            if (drawMode != DrawMode.Normal && DrawToolTipView != null)
            {
                //Calls the event -- here you can be creative
                DrawToolTipView(this, dte);
            }
            else
            {
                //The control will render itself -- we will do that in highest quality
                e.Graphics.TextRenderingHint = System.Drawing.Text.TextRenderingHint.ClearTypeGridFit;
                e.Graphics.SmoothingMode = SmoothingMode.HighQuality;

                //Now we can take advantage that we build an instance of DrawToolTipEventArgs
                dte.DrawBackground();

                //If the image is visible we have to place the text NEXT to the image 
                if (loadimg.Visible)
                {
                    TextRenderer.DrawText(e.Graphics, Text, Font,
                        new Rectangle(loadimg.Right + 4, 4, Width - loadimg.Right - 12, Height - 16),
                        color, TextFormatFlags.VerticalCenter);
                }
                else //Here we do not care and use all the space
                {
                    TextRenderer.DrawText(e.Graphics, Text, Font, new Rectangle(4, 4, Width - 8, Height - 16),
                        color,
                        TextFormatFlags.VerticalCenter | TextFormatFlags.HorizontalCenter | TextFormatFlags.WordBreak);
                }
            }
        }

        #endregion

        #region effect-enumeration

        /// <summary>
        /// Contains constants that are being used for setting the effect on ShowToolTip
        /// </summary>
        public enum Effect
        {
            Appear,
            Fade,
            Slide,
            SlideFade
        }

        #endregion
    }
}
