﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using SdlDotNet.Core;
using System.Drawing;
using System.Diagnostics;
using SdlDotNet.Audio;
using Animation;
using System.Reflection;

namespace TowerDefence
{
    public abstract class BaseObject
    {
        /// <summary>
        /// in units of the maze lattice constant in seconds
        /// </summary>
        [DescriptorProperty]
        public virtual float Speed
        {
            get { return 1.0f; }
        }

        /// <summary>
        /// Name of the object to be used in the UI
        /// </summary>
        public virtual string NickName
        {
            get { return String.Empty; }
        }

        public PointF MazeCoordinates { get; set; }

        private Direction direction;
        /// <summary>
        /// of view
        /// </summary>
        public Direction Direction
        {
            get { return direction; }
            set
            {
                if (direction != value)
                {
                    direction = value;
                    ChangeSpriteDirection();
                }
            }
        }

        /// <summary>
        /// position in screen coordinates
        /// </summary>
        public Point Position
        {
            get
            {
                return Sprite.Position;
            }
            set
            {
                Sprite.Position = value;
            }
        }

        /// <summary>
        /// Property to assess the 'goodness' of an object, e.g. for algorithms or pricing
        /// </summary>
        public virtual double Rating
        {
            get { return Speed; }
        }
        

        public bool Dead { get; set; }

        public virtual string Description { get { return @"abstract base class - so this description should show up nowhere in the game"; } }

        public BaseObject()
        {
            MazeCoordinates = Point.Empty;
            Dead = false;
            Direction = Direction.Right;
            LoadSprite();
        }

        /// <summary>
        /// Calculates distance in maze coordinates
        /// </summary>
        /// <param name="point">Point in maze coordinates to calculate distance to</param>
        /// <returns>Returns distance in maze coordinates</returns>
        public float DistanceTo(PointF point)
        {
            return point.Subtract(MazeCoordinates).Norm();
        }

        /// <summary>
        /// Updates on the SDLtick (game logic tick)
        /// </summary>
        /// <param name="args"></param>
        public virtual void Update(TickEventArgs args)
        {
            Sprite.Z = Position.Y;
        }

        #region sprite animations

        internal Sprite Sprite { get; private set; }

        protected virtual string SpriteFile { get { return "suns.bmp"; } }
        protected virtual int AnimationDelay { get { return 15; } }
        protected virtual int[] FramesPerDirection { get { return new int[] { 20 }; } }//new int[] { Right, Top, Left, Bottom }

        protected static List<Animation.Animation> animations;//only load animations once from file

        private FieldInfo getNewAnimation()
        {
            return this.GetType().getNewField("animations");
        }

        protected void LoadAnimation()
        {
            List<Animation.Animation> animations;
            if (FramesPerDirection.Length == 4)
            {
                animations = Animation.Animation.MultipleAnimationsFromFile(new TimeSpan(0, 0, 0, 0, AnimationDelay),
                    Helper.FindFile(SpriteFile), FramesPerDirection, new string[] { "Right", "Top", "Left", "Bottom" }, Color.White);
            }
            else
            {
                animations = new List<Animation.Animation>();
                animations.Add(Animation.Animation.AnimationFromFile(new TimeSpan(0, 0, 0, 0, AnimationDelay),
                    Helper.FindFile(SpriteFile), FramesPerDirection[0], "move", Color.White));
            }
            var newAnimations = getNewAnimation();
            newAnimations.SetValue(this, animations);
        }

        protected void LoadSprite()
        {
            FieldInfo newAnimations = getNewAnimation();
            if (newAnimations.GetValue(this) == null)
                LoadAnimation();
            Sprite = new Sprite(Point.Empty, newAnimations.GetValue(this) as List<Animation.Animation>);
        }

        protected void ChangeSpriteDirection()
        {
            if (Sprite != null)
                Sprite.SetAnimation(Direction.ToString());
        }

        #endregion
    }
}
