﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Drawing;
using Animation;

namespace TowerDefence
{
    /// <summary>
    /// Wrapper for the new GDI+ Animation::Stage
    /// </summary>
    class GDIView : IView
    {
        Stage stage;

        public Bitmap Screen { get { return stage.Image; } }

        public Size Size { get { return stage.Size; } }

        public Size MazeSize { get; private set; }

        List<BaseObject> objects;

        public event EventHandler<RefreshedEventArgs> Refreshed;
        
        public GDIView() : this(Size.Empty, Size.Empty) { }
        public GDIView(Size size, Size mazeSize)
        {
            MazeSize = mazeSize;
            objects = new List<BaseObject>();

            stage = new Stage(size, 25);
            setBackground(stage);

            stage.UpdatedEvent += (sender, e) =>
            {
                List<BaseObject> temp;
                lock (objects)
                {
                    temp = objects.ToList();
                }
                foreach (var item in temp)
                {
                    if (item.MazeCoordinates != Point.Empty)
                    {
                        item.Position = TransformMazeToScreen(item.MazeCoordinates).ToPoint();
                    }
                }
                if (Refreshed != null)
                {
                    Refreshed(this, new RefreshedEventArgs(Screen));
                }
            };
        }

        private void setBackground(Stage stage)
        {
            stage.Background = new Bitmap(Helper.FindFile("background.png"));

            using (var g = Graphics.FromImage(stage.Background))
            {
                //indicate Spawn
                float x = 1;
                float y = (MazeSize.Height / 2) * stage.Size.Height / MazeSize.Height + 1;
                float width = stage.Size.Width / MazeSize.Width - 1;
                float height = stage.Size.Height / MazeSize.Height - 1;
                SolidBrush brush = new SolidBrush(Color.FromArgb(70, Color.Red));
                g.FillRectangle(brush, x, y, width, height);

                //indicate Base
                x = (MazeSize.Width - 1) * stage.Size.Width / MazeSize.Width + 1;
                brush = new SolidBrush(Color.FromArgb(70, Color.Green));
                g.FillRectangle(brush, x, y, width, height);

                //Draw Grid
                brush = new SolidBrush(Color.FromArgb(3, Color.Black));
                for (int i = 1; i < MazeSize.Width; i++)
                {
                    for (int j = 1; j < MazeSize.Height; j++)
                    {
                        x = i * stage.Size.Width / MazeSize.Width;
                        y = j * stage.Size.Height / MazeSize.Height;
                        g.FillRectangle(brush, x, 0, 1, stage.Size.Height);
                        g.FillRectangle(brush, 0, y, stage.Size.Width, 1);
                    }
                }
            }
        }

        public void Start()
        {
            stage.Start();
        }

        public void Add(BaseObject baseObject)
        {
            lock (objects)
                objects.Add(baseObject);
            stage.Add(baseObject.Sprite);
        }
        public void Add(IEnumerable<BaseObject> baseObjects)
        {
            foreach (var item in baseObjects)
                Add(item);
        }

        public void Remove(BaseObject unit)
        {
            lock (objects)
                objects.Remove(unit);
            stage.Remove(unit.Sprite);
        }

        public PointF TransformMazeToScreen(PointF coordinates)
        {
            if (float.IsNaN(coordinates.X + coordinates.Y))
                return new PointF(-1, -1);
            else
                return new PointF(coordinates.X * Size.Width / MazeSize.Width + Size.Width / (2 * MazeSize.Width),
                    coordinates.Y * Size.Height / MazeSize.Height + Size.Height / (2 * MazeSize.Height));
        }

        public PointF TransformScreenToMaze(PointF coordinates)
        {
            if (float.IsNaN(coordinates.X + coordinates.Y))
                return new PointF(-1, -1);
            else
                return new PointF((coordinates.X - Size.Width / (2 * MazeSize.Width)) / Size.Width * MazeSize.Width,
                    (coordinates.Y - Size.Height / (2 * MazeSize.Height)) / Size.Height * MazeSize.Height);
        }

        public void Dispose()
        {
            stage.Dispose();
        }
    }
}
