﻿using System;
using System.Collections.Generic;
using System.Drawing;
using System.Linq;
using System.Text;
using Algorithms;

namespace TowerDefence
{
    public class Maze : IMaze
    {
        //Wegfindungsobjekt
        PathFinder pf;

        // PahtFinder pf wird mit grid initiallisiert. Dort werden Hindernisse gespeichert.
        byte[,] grid;

        Point start;
        Point end;

        //Liste bereits berechneter paths
        Dictionary<MazeStrategy, List<Point>> paths;

        //Dimension des diskretisierten Spielfeldes
        Size gridDim;
        public Size GridDim
        {
            get { return gridDim; }
        }

        //Setze den Start der Einheiten und das Ziel immer auf "ganz links und mittig" bzw. "ganz rechts und mittig".
        public Maze(Size _gridDim)
            : this(_gridDim, new Point(0, _gridDim.Height / 2), new Point(_gridDim.Width - 1, _gridDim.Height / 2))
        {
        }
        public Maze(Size _gridDim, Point _start, Point _end)
        {
            gridDim = _gridDim;

            if (!isPointInGrid(_start))
                throw new MazeException("Starting point isn't in the grid! ");
            if (!isPointInGrid(_end))
                throw new MazeException("Starting point isn't in the grid! ");

            start = _start;
            end = _end;
            grid = new byte[GridDim.Width + 1, GridDim.Height + 1];

            //Erstelle gird mit richtiger Dimension:
            for (int i = 0; i < GridDim.Width + 1; i++)
            {
                for (int j = 0; j < GridDim.Height + 1; j++)
                {
                    grid[i, j] = 1;
                }
            }

            //
            pf = new PathFinder(grid);

            paths = new Dictionary<MazeStrategy, List<Point>>();
        }

        //Methode welche die Gültigkeit eines Punktes prürft.
        bool isPointInGrid(Point p)
        {

            if (p.X > GridDim.Width - 1 || p.X < 0)
                return false;

            if (p.Y > GridDim.Height - 1 || p.Y < 0)
                return false;

            return true;

        }

        //Baue auf Punkt point einen Turm
        public bool TryBuild(Point point)
        {
            if (grid[point.X, point.Y] == 0)
                return false; //Hier steht schon ein Turm        

            if (point.X == this.gridDim.Width - 1 || point.X == 0) //Man darf nicht auf die letzte/erste Spalte bauen.
                return false;

            grid[point.X, point.Y] = 0; //Setze Tower;         

            pf.Formula = HeuristicFormula.Euclidean;
            pf.Diagonals = false;

            if (pf.FindPath(start, end) != null) //Der Tower darf gesetzt werden
            {
                paths.Clear();//Alte Pfade für ungültig erklären, da neues Objekt gebaut wurde.
                return true;
            }
            else
            {
                grid[point.X, point.Y] = 1; //Es wurde zugebaut... entferne Tower.
                return false;
            }
        }

        //Darf ein Turm am Punkt point abgebaut werden
        public bool TryRemove(Point point)
        {
            if (grid[point.X, point.Y] != 1)
            {
                grid[point.X, point.Y] = 1;
                paths.Clear();//Alte Pfade für ungültig erklären, da Objekt entfernt wurde.
                return true;
            }
            else
                return false;
        }
        
        //Finde den Weg durch die Türme unter Verwendung der Wegsuchstrategie MazeStrategy ms
        public List<Point> GetPath(MazeStrategy ms)
        {
            List<Point> ThePath = new List<Point>();

            if (!paths.ContainsKey(ms))
                calculate(ms);
            
            ThePath = paths[ms];

            return ThePath;
        }

        //Für den Luftweg muss niemand suchen :-)
        private List<Point> GetPathAir()
        {
            List<Point> AirPath = new List<Point>();

            for (int i = start.X; i < end.X + 1; i++)
                AirPath.Add(new Point(i, start.Y));

            return AirPath;
        }

        //Hier wird der Weg berechnet ... eben unter Verwendung von MazeStrategy ms und in ThePath gespeichert.
        private void calculate(MazeStrategy ms)
        {
            List<Point> ThePath;

            if (ms == MazeStrategy.Air)
            {
                ThePath = GetPathAir();
            }
            else
            {
                pf.Formula = (HeuristicFormula)ms;
                pf.Diagonals = false;

                var n = pf.FindPath(start, end);

                if (n == null)
                    throw new MazeException("Can't find the path!");

                ThePath = new List<Point>();

                foreach (var item in n)
                    ThePath.Add(new Point(item.X, item.Y));
            }

            paths.Add(ms, ThePath);
        }

        
        //Debugging Methode zur Ausgabe von Wegen und Türmen.
        public static void PrintPath(Maze maze, List<Point> ThePath)
        {
            //Erstelle gird mit richtiger Dimension:
            char[,] helpGrid = new char[maze.gridDim.Width, maze.gridDim.Height];
            for (int i = 0; i < maze.gridDim.Width; i++)
            {
                for (int j = 0; j < maze.gridDim.Height; j++)
                {
                    helpGrid[i, j] = maze.grid[i, j] == 0 ? (char)30 : '*';
                }
            }

            foreach (var item in ThePath)
            {
                helpGrid[item.X, item.Y] = (char)4;
            }

            helpGrid[maze.start.X, maze.start.Y] = 'S';
            helpGrid[ThePath.Last().X, ThePath.Last().Y] = 'E';

            for (int j = 0; j < maze.gridDim.Height; j++)
            {
                for (int i = 0; i < maze.gridDim.Width; i++)
                {

                    Console.OutputEncoding = Encoding.ASCII;
                    Console.Write(helpGrid[i, j]);
                }
                Console.Write("\r\n");
            }
        }
    }

    
    public class MazeException : Exception
    {
        public MazeException(String Message) : base(Message) { }
    }
}

