#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <string.h>
#include <omp.h>
#include "random.h"
#include "analysis.h"
#include "array.h"

/* Play around with those constants */
#define ALPHA 0.1
#define NSWEEP 10000000
#define DSIGMA 1
#define FILENAME "statistic.dat"

/* Do not change those constants */
#define STRING_BUFFER 1024
#define TRUE 1
#define FALSE 0
#define LOWEST_ORDER 0
#define HIGHEST_ORDER 1
#define NBLOCK 100000
#define PI 3.14159265358979323846
#define VERSION "1.2.0"

/* Inline helper functions as macros */
#define MAX(x, y) ((x) < (y) ? (y) : (x))
#define MIN(x, y) ((x) < (y) ? (x) : (y))
#define ABS(x) (SIGN(x) * x)
#define SIGN(x) ((x) > 0 ? 1 : ((x) < 0 ? -1 : 0))

/* Type for statistics */
typedef struct {
	int run;
	int rejects;
	int steps;
	int sum;
	int* Z;
	double alpha;
	double dsigma;
	double average;
	double error;
	double jkerr;
	double bserr;
	double autocor;
} t_statistic;

typedef struct {
	double* chain;
	double storage;
	int index;
	int subidx;
} t_block;

/* Shared variables */
t_statistic* statistics;

/* Thread local variables */
int order;
int signum;
double x;
t_block block;

#ifdef _OPENMP
#pragma omp threadprivate(order, signum, x, block)
#endif

/* Prints the available command line options */
void help() {
	printf("\n");
	printf(" Feynman Diagram Sampling: Integration\n");
	printf(" Sample integrand f(x) = cos(x) * exp(-alpha * x)\n");
	printf(" (c) Florian Rappl, 2013.\n");
	printf(" University of Regensburg.\n");
#ifdef _OPENMP
	printf(" OpenMP enabled.\n");
#endif
	printf(" Version %s (%s, %s).\n", VERSION, __DATE__, __TIME__);
	printf("\n");
	printf(" Available command line options:\n");
	printf("\n");
	printf("  -dsigma D\n");
	printf("\t Sets the value for the constant 0-level (default: 1).\n");
	printf("  -alpha D\n");
	printf("\t Sets the value for the constant alpha (default: 0.1).\n");
	printf("  -ns N\n");
	printf("\t Sets the number of iterations in a sweep (default: 1e8).\n");
	printf("  -autocorrelation B\n");
	printf("\t Sets if autocorrelation should be printed (default: 0).\n");
#ifdef _OPENMP
	printf("  -np N\n");
	printf("\t Sets the number of threads to use (default: 1).\n");
#endif
	printf("  -file S\n");
	printf("\t Sets the name of the file for output (default: statistic.dat).\n");
	printf("  --help\n");
	printf("\t Is currently displayed.\n");
	printf("\n");
}

/* Initializes the given block by setting up the resources */
void init_block(t_block* block, int n) {
	block->chain   = malloc(n / NBLOCK * sizeof(double));
	block->index   = 0;
	block->storage = 0.0;
	block->subidx  = 0;
}

/* Destroys the given block by releasing the bound memory */
void destroy_block(t_block* block) {
	free(block->chain);
}

/* Updates the given block by adding the given value */
void update_block(t_block* block, double value) {
	block->storage += value;

	if (++(block->subidx) == NBLOCK) {
		block->subidx = 0;
		block->chain[block->index] = block->storage / (double)NBLOCK;
		block->storage = 0.0;
		++(block->index);
	}
}

/* Initializes the given statistic instance */
void init_statistic(t_statistic* stat, int run, double alpha, double dsigma) {
	stat->rejects = 0;
	stat->sum     = 0;
	stat->steps   = 0;
	stat->error   = 0.0;
	stat->jkerr   = 0.0;
	stat->bserr   = 0.0;
	stat->autocor = 0.0;
	stat->average = 0.0;
	stat->run     = run;
	stat->alpha   = alpha;
	stat->dsigma  = dsigma;
	stat->Z       = malloc((HIGHEST_ORDER + 1) * sizeof(int));

	for (int i = 0; i <= HIGHEST_ORDER; i++) {
		stat->Z[i] = 0;
	}
}

/* Copies the stastics from the source structure to the target structure */
void copy_statistic(t_statistic* source, t_statistic* target) {
	target->rejects = source->rejects;
	target->steps   = source->steps;
	target->sum     = source->sum;
	target->run     = source->run;
	target->average = source->average;
	target->error   = source->error;
	target->jkerr   = source->jkerr;
	target->bserr   = source->bserr;
	target->alpha   = source->alpha;
	target->dsigma  = source->dsigma;
	target->autocor = source->autocor;
	target->Z       = source->Z;
}

/* Integrand I(x) of I */
double f(t_statistic* stat, double x) {
	return cos(x) * exp(-stat->alpha * x);
}

/* Integral I = \int I(x) dx */
double I(t_statistic* stat) {
	double h  = (double)stat->sum;
	double z  = (double)stat->Z[0];
	double d0 = stat->dsigma;
	return d0 * h / MAX(z, 1.0);
}

/* Error on the integral I = \int I(x) dx */
double Ierr(t_statistic* stat) {
	double h  = stat->error * stat->Z[1];

	if (stat->autocor > 0.5)
		h = h * sqrt(2 * stat->autocor);

	double z  = (double)stat->Z[0];
	double d0 = stat->dsigma;
	return d0 * h / MAX(z, 1.0);
}

/* Compute the value of the diagram at order */
double level(t_statistic* stat, int order) {
	if (order == HIGHEST_ORDER) {
		return cos(x) / stat->alpha;
	} else if (order == LOWEST_ORDER) {
		return stat->dsigma;
	} else {
		printf("ERROR: Cannot compute the value for this level!\n");
		exit(0);
	}
}

/* Try a Metropolis update from srcOrder (source) to destOrder (destination) */
int update(t_statistic* stat, int srcOrder, int destOrder) {
	int newsign    = 0;
	double higher  = 0.0;
	double lower   = 0.0;
	double ratio   = 1.0;
	double coin    = randomize();
	int high_order = MAX(srcOrder, destOrder);
	int low_order  = MIN(srcOrder, destOrder);

	/* If we increase the order we have to generate new variables / values */
	if (destOrder == HIGHEST_ORDER) {
		x = exponential(0.0, stat->alpha);
	}

	/* Compute the diagram values */
	higher  = level(stat, high_order);
	lower   = level(stat, low_order);

	/* Compute the ratio and new sign value */
	if (destOrder == LOWEST_ORDER) {
		ratio   = ABS(lower / higher);
		newsign = SIGN(lower);
	} else {
		ratio   = ABS(higher / lower);
		newsign = SIGN(higher);
	}

	/* Accept the step ?! */
	if (coin <= ratio) {
		signum = newsign;
		order  = destOrder;
		return 1;
	}

	return 0;
}

/* Perform Metropolis update step */
void step(t_statistic* stat) {
	int accepted = FALSE;

	if (LOWEST_ORDER == order) {
		accepted = update(stat, order, order + 1);
	} else if (HIGHEST_ORDER == order) {
		accepted = update(stat, order, order - 1);
	} else {
		printf("ERROR: Should have never reached this step!\n");
		exit(0);
	}

	if (accepted == FALSE) {
		stat->rejects++;
	}
}

/* Measure histogram */
void measure(t_statistic* stat) {
	/* The non-physical diagram won't be measured */
	if (order == HIGHEST_ORDER) {
		stat->sum += signum;
		update_block(&block, signum);
	}

	stat->Z[order]++;
}

/* Print statistics */
void print(t_statistic* stat) {
	double accepted = (double)(stat->steps - stat->rejects);
	double rejected = (double)stat->rejects * 100.0 / (double)stat->steps;
	printf("Accepted diagram changes    =  %.0lf\n", accepted);
	printf("Rejected diagram changes    =  %.0lf%%\n", rejected);

	for (int i = 0; i <= HIGHEST_ORDER; i++) {
		printf("Normalization statistics %d  =  %d\n", i, stat->Z[i]);
	}

	printf("-------------------------------------\n");
}

/* Saves the autocorrelation for the current chain */
void save_autocorrelation(t_statistic* stat) {
	char filename[50];
	sprintf(filename, "autocor_%d.dat", stat->run);
	FILE* file = fopen(filename, "w");
	int n = block.index;
	double* rs = getAutocorrelation(block.chain, n);

	if (file == NULL) {
		printf("WARNING: Cannot read file %s.\n", filename);
		exit(0);
	}

	for (int i = 0; i < n; i++) {
		fprintf(file, "%d\t%lf\t%lf\n", i, block.chain[i], rs[i]);
	}

	fclose(file);
}

/* Writes the whole statistic to a file */
void save_to_file(int nstat, char* filename) {
	FILE* file = fopen(filename, "w");

	if (file == NULL) {
		printf("WARNING: Cannot read file %s.\n", filename);
		exit(0);
	}

	fprintf(file, "alpha     \tDsigma    \tNsweep    \tNrej      \tAutocor.  \tReal      \tIest      \tIerr      \tAverage   \tError     \tJackknife \tBootstrap \n");

	for (int i = 0; i < nstat; i++) {
		t_statistic* stat = statistics + i;
		double estimated = I(stat);
		double real = stat->alpha / (1.0 + stat->alpha * stat->alpha);
		double error = Ierr(stat);

		fprintf(file, "%lf \t%lf \t%d  \t%d  \t%lf \t%lf \t%lf \t%lf \t%lf \t%lf \t%lf \t%lf\n", 
			stat->alpha, stat->dsigma, stat->steps, stat->rejects, stat->autocor, real, estimated, error, stat->average, stat->error, stat->jkerr, stat->bserr);
	}

	fclose(file);
}

/* Analysis method */
void analysis(int autocor, t_statistic* stat) {
	if (autocor == TRUE) {
		save_autocorrelation(stat);
	}

	double average, error;
	int n = block.index;
	doAnalysis(block.chain, n, &average, &error);
	stat->average = average;
	stat->error   = error;
	stat->jkerr   = getJackknifeError(average, block.chain, n);
	stat->bserr   = getBootstrapError(average, block.chain, n, /* nSamples : */ 3 * n, /* nData : */ 10);
	stat->autocor = getAutocorrelationTime(block.chain, n);
}

/* Metropolis iteration */
void metropolis(int autocor, int run, unsigned long nsweep, double alpha, double dsigma) {
	t_statistic current;
	init_statistic(&current, run, alpha, dsigma);
	init_block(&block, nsweep);

	/* Initialize the global values */
	signum     = 0;
	order      = 0;

	printf("Starting Metropolis procedure (alpha = %lf, dsigma = %lf, nsweep = %ld) ...\n", alpha, dsigma, nsweep);

	for (unsigned long i = 0; i < nsweep; ++i) {
		measure(&current);
		step(&current);
		current.steps++;
	}

	print(&current);
	printf("Running analysis ... ");
	analysis(autocor, &current);
	printf("done!\n");
	copy_statistic(&current, statistics + run);
	destroy_block(&block);
}

/* Reads the list of arguments at the given index and returns an array */
t_array* read_list(int index, int argc, char* argv[]) {
	double values[100];
	int count = 0;
	t_array* array;

	while (index < argc) {
		double value = atof(argv[index++]);

		if (value == 0.0)
			break;

		values[count++] = value;
	}

	array = malloc(1 * sizeof(t_array));
	array_init(array, sizeof(double), count);

	for (int i = 0; i < count; i++) {
		array_setDouble(array, i, values[i]);
	}

	return array;
}

/* Creates an array with just one value, which is the given one */
t_array* set_default(double value) {
	t_array* array = malloc(1 * sizeof(t_array));
	array_init(array, sizeof(double), 1);
	array_setDouble(array, 0, value);
	return array;
}

/* Main entry point */
int main(int argc, char* argv[]) {
	char* file  = FILENAME;
	int autocor = FALSE;
	t_array* as = NULL;
	t_array* ds = NULL;
	t_array* ns = NULL;
#ifdef _OPENMP
	int np      = 1;
#endif

	/* Read command line arguments */
	if (argc > 1) {
		for (int i = 1; i < argc; i++) {
			if (strcmp(argv[i], "-dsigma") == 0) {
				ds = read_list(i + 1, argc, argv);
				i += ds->length;
			} else if (strcmp(argv[i], "-alpha") == 0) {
				as = read_list(i + 1, argc, argv);
				i += as->length;
			} else if (strcmp(argv[i], "-ns") == 0) {
				ns = read_list(i + 1, argc, argv);
				i += ns->length;
			} else if (strcmp(argv[i], "-autocorrelation") == 0) {
				autocor = atoi(argv[++i]);
#ifdef _OPENMP
			} else if (strcmp(argv[i], "-np") == 0) {
				np = atoi(argv[++i]);
#endif
			} else if (strcmp(argv[i], "-file") == 0) {
				file = argv[++i];
			} else if (strcmp(argv[i], "--help") == 0) {
				help();
				return 0;
			} else {
				printf("Unknown command line argument. Use --help for all available options.\n");
				return 1;
			}
		}
	}

	if (as == NULL) {
		as = set_default(ALPHA);
	}

	if (ds == NULL) {
		ds = set_default(DSIGMA);
	}

	if (ns == NULL) {
		ns = set_default(NSWEEP);
	}

	int nstat = as->length * ds->length * ns ->length;
	statistics = malloc(nstat * sizeof(t_statistic));

#ifdef _OPENMP
	omp_set_num_threads(np);

#pragma omp parallel for
#endif
	for (int i = 0; i < nstat; i++) {
		double value = array_getDouble(ns, i % ns->length);
		unsigned long long nsweep = (unsigned long long)value;
		double alpha = array_getDouble(as, (i / ns->length) % as->length);
		double dsigma = array_getDouble(ds, (i / (ns->length * as->length)) % ds->length);
		/* Start Metropolis algorithm / iteration */
		metropolis(autocor, i, nsweep, alpha, dsigma);
	}

	/* Save values */
	save_to_file(nstat, file);
	return 0;
}