﻿using Microsoft.Win32;
using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Windows;
using System.Windows.Controls;
using System.Windows.Input;
using System.Windows.Media;
using System.Windows.Shapes;

namespace LevelEditor.ViewModel
{
    class MainWindowViewModel : ViewModelBase
    {
        Int32 tileSize = 32;
        Boolean IsModified;
        Track _CurrentTrack;
        ObservableCollection<TileItemViewModel> _Tiles;
        TileItemViewModel _SelectedTile;
        Boolean _FirstTrackLoaded;
        Canvas myCanvas;// this will be overdone soon
        RelayCommand _OpenFileCommand;
        RelayCommand _SaveFileCommand;
        RelayCommand _NewFileCommand;

        #region Properties

        public Track CurrentTrack
        {
            get { return _CurrentTrack; }
            set { _CurrentTrack = value; RaisePropertyChanged(); }
        }

        public Boolean FirstTrackLoaded
        {
            get { return _FirstTrackLoaded; }
            set { _FirstTrackLoaded = value; RaisePropertyChanged(); }
        }

        public TileItemViewModel SelectedTile
        {
            get { return _SelectedTile; }
            set { _SelectedTile = value; RaisePropertyChanged(); }
        }

        public ObservableCollection<TileItemViewModel> Tiles
        {
            get { return _Tiles; }
        }

        public ICommand OpenFileCommand
        {
            get { return _OpenFileCommand; }
        }

        public ICommand SaveFileCommand
        {
            get { return _SaveFileCommand; }
        }

        public ICommand NewFileCommand
        {
            get { return _NewFileCommand; }
        }

        public HitTestResultCallback MyCallBack { get; set; }

        #endregion

        #region ctor

        public MainWindowViewModel(Canvas canvas)
        {
            myCanvas = canvas;
            FirstTrackLoaded = false;
            MyCallBack = new HitTestResultCallback(HitTestCallBack);

            _OpenFileCommand = new RelayCommand(_ => OpenFileButton_Click());
            _SaveFileCommand = new RelayCommand(_ => SaveFileButton_Click());
            _NewFileCommand = new RelayCommand(_ => NewFileButton_Click());

            _Tiles = new ObservableCollection<TileItemViewModel>();
            var availableGroundTypes = Enum.GetValues(typeof(GroundTypes));

            for (int i = 0; i < availableGroundTypes.Length; i++)
                _Tiles.Add(new TileItemViewModel(this, (GroundTypes)i));
        }

        #endregion

        public void SetToModified()
        {
            IsModified = true;
        }

        #region canvas handling


        private void LoadTrackIntoCanvas(Track track)
        {
            var tiles = track.GroundTiles;
            var width = tiles.Width;
            var height = tiles.Height;

            myCanvas.Children.Clear();

            for (int i = 0; i < width; i++)
            {
                for (int j = 0; j < height; j++)
                {
                    var current = new Rectangle();
                    current.Width = tileSize;
                    current.Height = tileSize;
                    current.Fill = tiles.Grid[i, j].ToBrush();
                    myCanvas.Children.Add(current);
                    Canvas.SetLeft(current, tileSize * j);
                    Canvas.SetTop(current, tileSize * i);
                }
            }

            FirstTrackLoaded = true;
        }

        private HitTestResultBehavior HitTestCallBack(HitTestResult result)
        {
            var intersectionResult = result as HitTestResult;
            if (intersectionResult != null)
            {
                var temp = intersectionResult.VisualHit as Rectangle;

                if (temp != null && SelectedTile != null)
                {
                    var tile = SelectedTile.GroundType;
                    temp.Fill = tile.ToBrush();

                    Int32 j = (Int32)Canvas.GetLeft(temp) / tileSize;
                    Int32 i = (Int32)Canvas.GetTop(temp) / tileSize;

                    _CurrentTrack.GroundTiles.Grid[i, j] = tile;

                }
            }

            return HitTestResultBehavior.Continue;
        }
        
        #endregion

        #region load track

        private async void OpenFileButton_Click()
        {
            if (IsModified)
            {
                var result = MessageBox.Show("There are unsaved changes. Do you want to save them?", "Warning", MessageBoxButton.YesNoCancel);

                if (result == MessageBoxResult.Yes)
                {
                    await SaveCurrentAsync();
                }
                else if (result == MessageBoxResult.Cancel)
                {
                    return;
                }
            }
            await LoadCurrentAsync();
            LoadTrackIntoCanvas(_CurrentTrack);
            IsModified = false;
            SelectedTile = null;
        }

        async Task LoadCurrentAsync()
        {
            var openDialog = new OpenFileDialog();
            openDialog.Filter = "Formula Sharp Tracks|*.track|All Files|*.*";
            var result = openDialog.ShowDialog();

            if (result.HasValue && result.Value)
            {
                using (var stream = openDialog.OpenFile())
                {
                    _CurrentTrack = await Track.LoadAsync(stream);
                }

            }
        }

        #endregion

        #region save track

        private async void SaveFileButton_Click()
        {
            await SaveCurrentAsync();
            IsModified = false;
        }

        async Task SaveCurrentAsync()
        {
            var saveDialog = new SaveFileDialog();
            saveDialog.Filter = "Formula Sharp Tracks|*.track|All Files|*.*";
            var result = saveDialog.ShowDialog();

            if (result.HasValue && result.Value)
            {
                using (var stream = saveDialog.OpenFile())
                {
                    await _CurrentTrack.SaveAsync(stream);
                }
            }
        }

        #endregion

        #region new track

        private async void NewFileButton_Click()
        {
            if (IsModified)
            {
                var result = MessageBox.Show("There are unsaved changes. Do you want to save them?", "Warning", MessageBoxButton.YesNoCancel);

                if (result == MessageBoxResult.Yes)
                {
                    await SaveCurrentAsync();
                }
                else if (result == MessageBoxResult.Cancel)
                {
                    return;
                }
            }

            var sizePicker = new PropertyPickerWindow();
            if (sizePicker.ShowDialog() == true)
            {
                var vm = sizePicker.DataContext as PropertyPickerWindowViewModel;
                if (vm != null)
                {
                    
                    var name = vm.SelectedName;
                    var grid = new GroundTypes[vm.TrackHeight, vm.TrackWidth];

                    var tiles = new GroundTileGrid(grid);
                    for (int i = 0; i < tiles.Width; i++)
                    {
                        for (int j = 0; j < tiles.Height; j++)
                        {
                            tiles.Grid[i, j] = vm.BaseTile;
                        }
                    }
                    _CurrentTrack = new Track(name,tiles);

                    LoadTrackIntoCanvas(_CurrentTrack);
                    IsModified = false;

                }
            }
        }

        #endregion

    }
}
